#include"sci.h"

void (*sci_callback)(unsigned char, unsigned char);  //This is a pointer to the function which 
                                                     //the user needs to pass in SCI_Init
                                                     //so that the required function is 
                                                     //called in interrupt subroutine

unsigned char* SCI0_Data;    //stores the data for SCI0 Tx and Rx in case of interrupt
unsigned char* SCI1_Data;    //stores the data for SCI1 Tx and Rx in case of interrupt
unsigned char* SCI2_Data;    //stores the data for SCI2 Tx and Rx in case of interrupt
unsigned char* SCI3_Data;    //stores the data for SCI3 Tx and Rx in case of interrupt

unsigned char SCI0_DataLen;  //stores the length of the data to be transmitted or read from SCI0
unsigned char SCI1_DataLen;  //stores the length of the data to be transmitted or read from SCI1
unsigned char SCI2_DataLen;  //stores the length of the data to be transmitted or read from SCI2
unsigned char SCI3_DataLen;  //stores the length of the data to be transmitted or read from SCI3

unsigned char* SCI_RxData;   //stores the recived array from any SCI port in case interrupt is disabled

/******************************************************************************/
/* FUNCTION          : SCI_Init                                               */
/******************************************************************************/
/* Abstract          : This function is used to Initialise the specific SCI   */
/*	                                                                          */
/*                                                                            */
/* Input Parameters  : SCI_Index(to select the SCI to be initialted)          */
/*                     0:SCI0 ,1:SCI1, 2:SCI2, 3:SCI3                         */
/*                     p function is used only in case of interrupts the user */
/*                     can pass the address of the function but he has to be  */
/*                     careful cause this function is an interrupt function   */                                                               
/*                                                                            */
/* Return Parameter  : None                                                   */
/*                                                                            */
/* Traceability Info :                                                        */
/******************************************************************************/
  
 void SCI_Init(unsigned char SCI_Index,void (*p)(unsigned char, unsigned char))
   {
    
    SCI *sci_ptr;
    SCGC4_MUXCTRL=1;                 //enables the clock to the pin mux module
    
    sci_callback = p; 
    switch(SCI_Index) 
    {
      case 0:
        {
          if(INTERNAL_PULL_UP_SCI0) 
          { 
            SCGC3_PTB = 1;            //enables the clock for PTB module
            PTBDD_PTBDD3 = 1;         //makes the PTB3 as output
            PTBPE_PTBPE3 = 1;         //enables the pull for PTB3
          }
          
          sci_ptr = (SCI*)SCI_0;      //points to SCI0
          SCGC1_SCI0 = 1;             //enables the clock to SCI0 module
          PTBPF2 = 0x44;              //pin B3 as TXD0 & pin B2 as RXD0
          break;
        }
      
      case 1: 
        {
          sci_ptr = (SCI*)SCI_1;      //points to SCI1
          SCGC1_SCI1 = 1;             //enables the clock to the SCI1 module
          
          if(SCI1_TxRx_PortC) 
          {  
            PTCPF3 = 0x22;              //pin C5 as TXD1 & pin C4 as RXD1
          }
          else 
          {  
            PTBPF1_B1 = 0b010;          //pin B1 as TXD1
            PTBPF1_B0 = 0b010;          //pin B0 as RXD1
          }
          break;
        }
      
      case 2:
        {
          sci_ptr = (SCI*)SCI_2;
          SCGC1_SCI2 = 1;            //enables the clock to the SCI2 module
          
          if(SCI2_TxRx_PortA)
          {  
            PTAPF3=0x22;                //pin A5 as TXD2 & pin A4 as RXD2
          }
          else
          {
            
            PTBPF4=0x22;               //pin B7 as TXD2 & pin B6 as RXD2
          }
          
          break;
        }
      
      case 3: 
        {
          sci_ptr = (SCI*)SCI_3;
          SCGC1_SCI3 = 1;            //enables the clock to the SCI3 module 
          
          if(SCI3_TxRx_PortC)
          {  
            PTCPF4=0x22;               //pin C6 as RXD3 & pin C7 as TXD3
          }
       
          else 
          {  
            PTGPF3_G5 = 0b010;         //pin G5 as TXD3
            PTGPF3_G4 = 0b010;         //pin G4 as RXD3
          }
          break;
        }
    }
    
    
    sci_ptr->SCIC1.Byte =0x00;    /* 
                                  LSB   PT (Parity type)
                                        PE (Parity bit enable)
                                        ILT(Idle line type select)
                                        WAKE (Reciever wakeup select)
                                        M(8/9 bit mode select)
                                        RSRC(Receiver source select)
                                        SCISWAI(SCI stops in stop mode)
                                        LOOPS(Loop / normal mode select)
                                   */    
    sci_ptr->SCIC2.Byte = 0x0C;  //enables the Tx and Rx
                                 /*     LSB   SBK(Send Break)
                                        RWU(Reciever wakeup control)
                                        RE(Reciever enable)
                                        TE(Transmitter enable)
                                        ILIE(Idle line interrupt enable)
                                        RIE(Reciever interrupt enable)
                                        TCIE(Transmission complete interrupt enable)
                                        TIE(Transmit interrupt enable)
                                  */
                                  
                                                                 
    sci_ptr->SCIC3.Byte = 0x00;  /*
                                   LSB   PEIE(Parity error interrupt enable)
                                         FEIE(Framing error interrupt enable)
                                         NEIE(Noise error interrupt enable)
                                         ORIE(Overrun interrupt enable)
                                         TXINV(Transmit data inversion)
                                         TXDIR(TxD pin direction in single wire mode)
                                         T8(9th data bit for transmitter)
                                         R8(9th data bit for receiver)
    
                                   */ 
                                                            
    
    SCI_Set_BaudRate_38400(SCI_Index);  //setting the baud rate to 38400
   }
       
 
/******************************************************************************/
/* FUNCTION          : SCI_Set_BaudRate                                       */
/******************************************************************************/
/* Abstract          : This function is used to set the required baud rate    */
/*                     of the specific SCI                                    */
/*	                                                                          */
/*                                                                            */
/* Input Parameters  : SCI_Index                                              */
/*                     0:SCI0 ,1:SCI1, 2:SCI2, 3:SCI3                         */
/*                     Baud_Rate (baud rate to be set)                        */                                                                                                                                                                       
/*                                                                            */
/* Return Parameter  : None                                                   */
/*                                                                            */
/* Traceability Info :                                                        */
/******************************************************************************/


void SCI_Set_BaudRate(unsigned char SCI_Index,unsigned int Baud_Rate) 
{
  unsigned int temp;
  temp = BUS_CLK/Baud_Rate;            //calculates the baud rate divisor
  temp = temp/16;  
  switch(SCI_Index) 
  { 
    case 0:
    SCI0BDL=temp;             //Higher byte of the baud rate divisor SCI0
    SCI0BDH=(temp>>8);        //lower byte of the baud rate divisor SCI0
    break;
    
    case 1:
    SCI1BDL=temp;             //Higher byte of the baud rate divisor SCI1
    SCI1BDH=(temp>>8);        //lower byte of the baud rate divisor  SCI1
    break;                    
    
    case 2:
    SCI2BDL=temp;             //Higher byte of the baud rate divisor SCI2
    SCI2BDH=(temp>>8);        //lower byte of the baud rate divisor  SCI2
    break;
    
    case 3:
    SCI3BDL=temp;             //Higher byte of the baud rate divisor SCI3
    SCI3BDH=(temp>>8);        //lower byte of the baud rate divisor SCI3
    break;
    
  }
  
}


/******************************************************************************/
/* FUNCTION          : SCI_PutChar                                            */
/******************************************************************************/
/* Abstract          : This function sends 1 byte data to the                 */
/*                      specific SCI port                                     */					                            
/*                                                                            */
/* Input Parameters  : SCI_Index                                              */
/*                     0:SCI0 ,1:SCI1, 2:SCI2, 3:SCI3                         */
/*                     Send_Data (data to be sent)                            */                                                                                                                                                                   
/*                                                                            */
/* Return Parameter  : None                                                   */
/*                                                                            */
/* Traceability Info :                                                        */
/******************************************************************************/

 void SCI_PutChar(unsigned char SCI_Index,unsigned char Send_Data)
{
  unsigned char temp;
  SCI *sci_ptr;
  switch(SCI_Index) 
  {
    case 0:
    sci_ptr = (SCI*)SCI_0;              //points to SCI0
    break;
    
    case 1:
    sci_ptr = (SCI*)SCI_1;              //points to SCI1
    break;
    
    case 2:
    sci_ptr = (SCI*)SCI_2;              //points to SCI2
    break;
    
    case 3:
    sci_ptr = (SCI*)SCI_3;              //points to SCI3
  }

 
   while((sci_ptr->SCIC2.Bits.TIE)||(sci_ptr->SCIC2.Bits.RIE));       //waiting if some interrupt is executed
   while(!(sci_ptr->SCIS1.Bits.TDRE));   //waiting for the flag to be set 
   temp = sci_ptr->SCIS1.Byte;
  
   sci_ptr->SCID  = Send_Data;      //sends data to the data register of SCI
 
 
}

/******************************************************************************/
/* FUNCTION          : TERMIO_PutChar                                         */
/******************************************************************************/
/* Abstract          : This function sends 1 byte data to the                 */
/*                      specific SCI port defined as UART_SEL                 */					                            
/*                                                                            */
/* Input Parameters  : Character (data to be sent)                            */
/*                                                                            */
/* Return Parameter  : None                                                   */
/*                                                                            */
/* Traceability Info :                                                        */
/******************************************************************************/

void TERMIO_PutChar(unsigned char character)
{
  SCI_PutChar(UART_SEL, character);   //sends data to the port as defined in UART_SEL
}



/******************************************************************************/
/* FUNCTION          : SCI_DataTransferString                                 */
/******************************************************************************/
/* Abstract          : This function sends an array of data to the SCI port   */
/*					           depending upon the selection of SCI module             */
/*                                                                            */
/* Input Parameters  : SCI_Index(0:SCI0 ,1:SCI1, 2:SCI2, 3:SCI3)              */
/*                     interrupt_enable(1:enable the tx interrupt 0:disabled  */
/*                     array(data array to be transferred)                    */
/*                     length(length of the array to be transferred)          */
/* Return Parameter  : None                                                   */
/*                                                                            */
/* Traceability Info :                                                        */
/******************************************************************************/

void SCI_SendArray(unsigned char SCI_Index,unsigned char interrupt_enable,
                             unsigned char* array,unsigned char length) 
{
   unsigned int i =0;
   if(interrupt_enable) 
   {
      SCI *sci_ptr;
      switch(SCI_Index) 
      {
        case 0:
          sci_ptr = (SCI*)SCI_0;          //points to SCI0
          while((sci_ptr->SCIC2.Bits.TIE)||(sci_ptr->SCIC2.Bits.RIE));     //if any pending interrupt is getting executed
          SCI0_Data = array;            //stores the array in a global variable SCI0_TxData
          SCI0_DataLen = length;        //stores the length of the array in global variable SCI0_TxDataLen
        break;
    
        case 1:
          sci_ptr = (SCI*)SCI_1;          //points to SCI1
          while((sci_ptr->SCIC2.Bits.TIE)||(sci_ptr->SCIC2.Bits.RIE));
          SCI1_Data = array;            //stores the array in a global variable SCI1_TxData
          SCI1_DataLen = length;        //stores the length of the array in global variable SCI0_TxDataLen
        break;
    
        case 2:
          sci_ptr = (SCI*)SCI_2;          //points to SCI2
          while((sci_ptr->SCIC2.Bits.TIE)||(sci_ptr->SCIC2.Bits.RIE));
          SCI2_Data = array;            //stores the array in a global variable SCI2_TxData
          SCI2_DataLen = length;        //stores the length of the array in global variable SCI0_TxDataLen
        break;
    
        case 3:
          sci_ptr = (SCI*)SCI_3;          //points to SCI3
          while((sci_ptr->SCIC2.Bits.TIE)||(sci_ptr->SCIC2.Bits.RIE));
          SCI3_Data = array;            //stores the array in a global variable SCI3_TxData
          SCI3_DataLen = length;        //stores the length of the array in global variable SCI0_TxDataLen
      }
    
   
   sci_ptr->SCIC2.Bits.TIE = 1;        //enables the Transmit interrupt
    
   }
   
  else  //if interrupt is disabled
   {
    while(i<=length) 
    {
      SCI_PutChar(SCI_Index,array[i]);     //transfers the array to the SCI port specified
     i++;
    }
   }
}


 /******************************************************************************/
/* FUNCTION          :SCI_GetChar                                             */
/******************************************************************************/
/* Abstract          : This function is used to recieve char data from        */
/*					           SCI port                                               */
/*                                                                            */
/* Input Parameters  : SCI_Index                                              */
/*                                                                            */
/* Return Parameter  : None                                                   */
/*                                                                            */
/* Traceability Info :                                                        */
/******************************************************************************/ 
 
  unsigned char SCI_GetChar(unsigned char SCI_Index)
  { 
    unsigned char data;
    unsigned char dummy;
    SCI *sci_ptr;
   switch(SCI_Index) 
    {
     case 0:
     sci_ptr = (SCI*)SCI_0;
     break;
    
     case 1:
     sci_ptr = (SCI*)SCI_1;
     break;
    
     case 2:
     sci_ptr = (SCI*)SCI_2;
     break;
     
     case 3:
     sci_ptr = (SCI*)SCI_3; 
    }
    
    while((sci_ptr->SCIC2.Bits.TIE)||(sci_ptr->SCIC2.Bits.RIE));
    while(!(sci_ptr->SCIS1.Bits.RDRF));
    dummy = sci_ptr->SCIS1.Byte;
    data = sci_ptr->SCID;
    return data;
  }


/******************************************************************************/
/* FUNCTION          :SCI_GetArray                                            */
/******************************************************************************/
/* Abstract          : This function is used to recieve char data from        */
/*					           SCI port                                               */
/*                                                                            */
/* Input Parameters  : SCI_Index                                              */
/*                     interrupt_enable(1:enable the tx interrupt 0:disabled  */
/*                     length( length of the data to be recieved)             */
/*                                                                            */
/* Return Parameter  : None                                                   */
/*                                                                            */
/* Traceability Info :                                                        */
/******************************************************************************/ 


 void SCI_GetArray(unsigned char SCI_Index,unsigned char interrupt_enable,unsigned char* array,unsigned char length) 
 {
  unsigned char i=0;
  if(interrupt_enable)
   {
     SCI *sci_ptr;
     switch(SCI_Index) 
      {
        case 0:
        sci_ptr = (SCI*)SCI_0;
        while((sci_ptr->SCIC2.Bits.RIE)||(sci_ptr->SCIC2.Bits.TIE)); //waits if some other interrupt is getting executed
        SCI0_DataLen = length;
        SCI0_Data = array;
        break;
    
        case 1:
        sci_ptr = (SCI*)SCI_1;
        while((sci_ptr->SCIC2.Bits.RIE)||(sci_ptr->SCIC2.Bits.TIE)); //waits if some other interrupt is getting executed
        SCI1_DataLen = length;
        SCI1_Data = array;
        break;
    
        case 2:
        sci_ptr = (SCI*)SCI_2;
        while((sci_ptr->SCIC2.Bits.RIE)||(sci_ptr->SCIC2.Bits.TIE)); //waits if some other interrupt is getting executed
        SCI2_DataLen = length;
        SCI2_Data = array;
        break;
     
        case 3:
        sci_ptr = (SCI*)SCI_3;
        while((sci_ptr->SCIC2.Bits.RIE)||(sci_ptr->SCIC2.Bits.TIE)); //waits if some other interrupt is getting executed
        SCI3_DataLen = length;
        SCI3_Data = array; 
      }
      
     sci_ptr->SCIC2.Bits.RIE = 1;        //enables the Rx interrupt
   }
  
  else    //if interrupt is disabled
  {
    while(i<length) 
    {
     array[i] = SCI_GetChar(SCI_Index);  //gets the array without the interrupt
     i++;
    }
  }
  
 }
/******************************************************************************/
/* FUNCTION          : TERMIO_GetChar                                         */
/******************************************************************************/
/* Abstract          : This function reads 1 byte data from the               */
/*                      specific SCI port defined as UART_SEL                 */					                            
/*                                                                            */
/* Input Parameters  : Character (data to be sent)                             */
/*                                                                            */
/* Return Parameter  : None                                                   */
/*                                                                            */
/* Traceability Info :                                                        */
/******************************************************************************/

unsigned char TERMIO_GetChar()
{
  return (SCI_GetChar(UART_SEL));   //sends data to the port as defined in UART_SEL
}

    
 //interrupt subroutine
 void interrupt VectorNumber_Vsci0tx SCITX0_ISR() 
 {
  char dummy;
  dummy = SCI0S1;      
  SCI0D = *SCI0_Data;  //clears the TDRE interrupt and sends the data
  SCI0_Data++;
  SCI0_DataLen--;
  if(sci_callback != 0)
   (*sci_callback)(SCI0_TX,0);//sends the data to a function pointed by sci_callback

  if(SCI0_DataLen == 0) 
  {  
   SCI0C2_TIE = 0;    //disables the interrupt when the entire data array is transmitted
  }
  
 }
 
 void interrupt VectorNumber_Vsci0rx SCIRX0_ISR()
  {
    char dummy;
    dummy = SCI0S1;
   *SCI0_Data = SCI0D; //clear the RDRF interrupt and reads the data
    SCI0_Data++;
    SCI0_DataLen--;
    
    if(sci_callback != 0)
     (*sci_callback)(SCI0_RX,*SCI0_Data);//sends the data to a function pointed by sci_callback
   
   if(SCI0_DataLen == 0) 
    {
     SCI0C2_RIE = 0;
    }
  }


 void interrupt VectorNumber_Vsci1tx SCITX1_ISR() 
 {
  char dummy;
  dummy = SCI1S1;      //clears the TDRE interrupt and sends the data
  SCI1D = *SCI1_Data;
  SCI1_Data++;
  SCI1_DataLen--;
  if(sci_callback != 0)
   (*sci_callback)(SCI1_TX,0);//sends the data to a function pointed by sci_callback

  if(SCI1_DataLen==0) 
  {  
   SCI1C2_TIE = 0;     //disables the interrupt when the entire data array is transmitted
  }
 }
 
 void interrupt VectorNumber_Vsci1rx SCIRX1_ISR()
  {
    char dummy;
    dummy = SCI1S1;
   *SCI1_Data = SCI1D; //clear the RDRF interrupt and reads the data
    SCI1_Data++;
    SCI1_DataLen--;
    
    if(sci_callback != 0)
   (*sci_callback)(SCI1_RX,*SCI1_Data);//sends the data to a function pointed by sci_callback
   
   if(SCI1_DataLen == 0) 
    {
     SCI1C2_RIE = 0;
    }  
  } 

 void interrupt VectorNumber_Vsci2tx SCITX2_ISR() 
  {
   char dummy;
   dummy = SCI2S1;      //clears the TDRE interrupt and sends the data
   SCI2D = *SCI2_Data;
   SCI2_Data++;
   SCI2_DataLen--;
   if(sci_callback != 0)
   (*sci_callback)(SCI2_TX,0);//sends the data to a function pointed by sci_callback

   if(SCI2_DataLen == 0) 
   {  
    SCI2C2_TIE = 0;      //disables the interrupt when the entire data array is transmitted
   }
  }
 
 void interrupt VectorNumber_Vsci2rx SCIRX2_ISR()
  {
    char dummy;
    dummy = SCI2S1;
   *SCI2_Data = SCI2D; //clear the RDRF interrupt and reads the data
    SCI2_Data++;
    SCI2_DataLen--;
    
    if(sci_callback != 0)
   (*sci_callback)(SCI2_RX,*SCI2_Data);//sends the data to a function pointed by sci_callback
   
    if(SCI2_DataLen == 0) 
    {
     SCI2C2_RIE = 0;
    }  
  }

 void interrupt VectorNumber_Vsci3tx SCITX3_ISR() 
 {
  char dummy;
  dummy = SCI3S1;      //clears the TDRE interrupt and sends the data
  SCI3D = *SCI3_Data;
  SCI3_Data++;
  SCI3_DataLen--;
  if(sci_callback != 0)
   (*sci_callback)(SCI3_TX,0);//sends the data to a function pointed by sci_callback

  if(SCI3_DataLen == 0) 
  {  
   SCI3C2_TIE = 0;      //disables the interrupt when the entire data array is transmitted
  }
 }
 
 void interrupt VectorNumber_Vsci3rx SCIRX3_ISR()
  { 
    char dummy;
    dummy = SCI3S1;
   *SCI3_Data = SCI3D;    //clear the RDRF interrupt and reads the data
    SCI3_Data++;
    SCI3_DataLen--;
    
  if(sci_callback != 0)
   (*sci_callback)(SCI3_RX,*SCI3_Data);//sends the data to a function pointed by sci_callback
   
   if(SCI3_DataLen == 0) 
    {
     SCI3C2_RIE = 0;
    }
  }

 


 
